import re
from pathlib import Path
from shutil import Error as ShutilError
from shutil import rmtree

import export_corese
from common import *

log = get_logger("export")


def create_file(root: Path, path: Path, body: str):
    # Remove root of activity path if present, we need a relative path
    path = path.relative_to(path.root)
    path = root / path
    # Create parent dir if needed
    path.parent.mkdir(parents=True, exist_ok=True)
    # Create and write file
    log.debug(f"File {path} ->\n{body}")
    with open(path, "w") as f:
        f.write(body)


def safe_to_remove(path: Path) -> bool:
    """Simple safeguard before `rm -rf`ing a directory, that checks if the path
    seems reasonable to delete."""
    if str(path).count("macao") == 0:
        log.warning("'{path}' may be unsafe to remove: does not contain 'macao'.")
        return False
    is_git = False
    while path.parent != path:
        path = path.parent
        if (path / ".git").is_dir():
            is_git = True
            break
    if not is_git:
        log.warning("'{path}' may be unsafe to remove: not a Git repository.")
    return is_git


def main():
    # Prepare the content root dir: delete and recreate if it exists
    root = Path(NEW_CONTENT_ROOT)
    if root.exists() and safe_to_remove(root):
        try:
            rmtree(root)
        except ShutilError:
            log.error("Failed to delete old content before export", exc_info=True)
    root.mkdir(parents=True, exist_ok=True)

    # ==> Apply STTL transformations
    template_output = export_corese.apply_templates()
    # ==> Split the result on the start-of-file markers
    regex = re.compile(r"^####### *?(\S+)\n", re.MULTILINE)
    parts = regex.split(template_output)
    # The parts list starts with the first non-delimiter string (which may be
    # empty), followed by each capturing group result of the delimiter, then
    # the second non-delimiter, and so on.
    # With a pattern that has 1 group, its results are on odd-numbered indices.
    for i in range(1, len(parts), 2):
        file_path = parts[i]
        file_content = parts[i + 1]
        create_file(root, Path(file_path), file_content)
    log.info(f"Found {len(parts)} parts => {(len(parts)-1)/2} files.")


if __name__ == "__main__":
    main()
