#!/usr/bin/env python3
"""
Compares two RDF graphs given as separate files, indicating if they're
isomorphic, and their differences otherwise.
Implemented from https://rdflib.readthedocs.io/en/stable/apidocs/rdflib.html#module-rdflib.compare
"""
from sys import argv, stderr

from rdflib import Graph
from rdflib.compare import graph_diff, to_isomorphic


def dump_nt_sorted(g: Graph):
    for l in sorted(g.serialize(format="nt").splitlines()):
        if l:
            print("\t" + l)


def main():
    if len(argv) < 3:
        print(f"Usage: {argv[0]} <first_graph> <second_graph>", file=stderr)
        exit()

    g1 = Graph().parse(argv[1])
    g2 = Graph().parse(argv[2])
    iso1 = to_isomorphic(g1)
    iso2 = to_isomorphic(g2)
    if iso1 == iso2:
        print("Graphs are isomorphic.")
    else:
        print("Graphs are NOT isomorphic.")
        in_both, in_first, in_second = graph_diff(iso1, iso2)
        print("In both:")
        dump_nt_sorted(in_both)
        print("In first:")
        dump_nt_sorted(in_first)
        print("In second:")
        dump_nt_sorted(in_second)


if __name__ == "__main__":
    main()
